package de.ullisroboterseite.ursai2bytearray;

// Autor: https://UllisRoboterSeite.de

// Doku:  https://UllisRoboterSeite.de/android-AI2-ByteArray.html
//
// Version 1.0 (2021-04-02)
// -------------------------
// - Basis-Version
//
//
// Version 1.1 (2021-05-21)
// -------------------------
// - Fehler bei  ReadWord, ReadDWord und ToHex behoben
//
//
// Version 1.2 (2021-12-13)
// -------------------------
// - Methode toString hat eine Methode benutzt, die erst mit JAVA 8 zur Verfügung steht.
//
//
// Version 1.3 (2022-05-13)
// -------------------------
// - Methoden WriteToFile, ReadFromFile und DeleteFile hinzugefügt.
// - Methoden AddUTF8String, AddASCIIString, ReadUTF8String, ReadASCIIString hinzugefügt.
// - Methode GetUTF8ByteSize hinzugefügt.
//
// Version 1.4 (2024-03-18)
// -------------------------
// - Methoden WriteToFileSync, ReadFromFileSync hinzugefügt.
// - ReadFromFile hatte das Ereignis AfterFileRead nicht ausgelöst.
//
// Version 1.5 (2024-07-24)
// - Methode CRC32, SetWord und SetDword hinzugefügt
// - Die Einstellung der Eigenschaft Base wurde bei ReadIndex nicht berücksichtigt.
//
// Version 1.6 (2025-02-17)
// - Methoden ReadUTF8StringUntil, ReadASCIIStringUntil und RemoveBytes hinzugefügt.

import java.util.*;
import java.lang.*;

import com.google.appinventor.components.annotations.*;
import com.google.appinventor.components.common.*;
import com.google.appinventor.components.runtime.*;

import com.google.appinventor.components.runtime.util.*;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.zip.CRC32;
import java.util.zip.Checksum;

import com.google.appinventor.components.runtime.errors.StopBlocksExecution;

import android.util.Log;

@DesignerComponent(version = 1, //
        versionName = UrsAI2ByteArray.VersionName, //
        dateBuilt = UrsAI2ByteArray.dateBuilt, //
        description = "AI2 extension block to store a byte array.", //
        category = com.google.appinventor.components.common.ComponentCategory.EXTENSION, //
        nonVisible = true, //
        helpUrl = "https://UllisRoboterSeite.de/android-AI2-ByteArray.html", //
        iconName = "aiwebres/icon.png")
@SimpleObject(external = true)
@UsesPermissions(permissionNames = "android.permission.WRITE_EXTERNAL_STORAGE, android.permission.READ_EXTERNAL_STORAGE")
public class UrsAI2ByteArray extends AndroidNonvisibleComponent {
    static final String LOG_TAG = "BYTE";
    static final String VersionName = "1.6.0";
    static final String dateBuilt = "2025-02-17";

    public ArrayList<Byte> bytes = new ArrayList<>();
    boolean msbFirst = true;
    int base = 1;
    String prefix = "0x";
    int readIndex = 0; // Intern ist der Index 0-basiert. Extern muss der Wert von Base berücksichtigt
                       // werden.
    private FileScope scope = FileScope.App;

    public UrsAI2ByteArray(ComponentContainer container) {
        super(container.$form());
    }

    @SimpleProperty(description = "Returns the component's version name.")
    public String Version() {
        return VersionName;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_BOOLEAN, defaultValue = "True")
    @SimpleProperty(description = "Byte order for words and integers.")
    public void MsbFirst(boolean value) {
        msbFirst = value;
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Byte order for words and integers.")
    public boolean MsbFirst() {
        return msbFirst;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "0x")
    @SimpleProperty(description = "Prefix for hexadecimal outputs.")
    public void HexPrefix(String value) {
        prefix = value;
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Prefix for hexadecimal outputs.")
    public String HexPrefix() {
        return prefix;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_NON_NEGATIVE_INTEGER, defaultValue = "1")
    @SimpleProperty(description = "Index of first element of the array (0 or 1).")
    public void Base(int value) {
        if (value > 1)
            value = 1;
        if (value < 0)
            value = 0;
        base = value;
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Index of first element of the array (0 or 1).")
    public int Base() {
        return base;
    }

    @SimpleProperty(description = "Index of the element that is read next.")
    public int ReadIndex() {
        return readIndex + base;
    }

    @SimpleProperty(description = "Index of the element that is read next.")
    public void ReadIndex(int Value) {
        if (Value < 0)
            Value = 0;
        readIndex = Value - base;
    }

    @SimpleProperty(description = "Returns the number of bytes that can be read.")
    public int Available() {
        int result = bytes.size() - readIndex;
        if (result < 0)
            result = 0;
        return result;
    }

    @SimpleProperty(description = "Returns the number of stored bytes.")
    public int Size() {
        return bytes.size();
    }

    @SimpleProperty(description = "Returns the CRC32 of stored bytes.")
    public long CRC32() {
        Checksum crc32 = new CRC32();
        crc32.update(toByteArray(), 0, bytes.size());
        return crc32.getValue();
    }

    @SimpleFunction(description = "Removes all entries.")
    public void Clear() {
        bytes = new ArrayList<>();
        readIndex = 0;
    }

    @SimpleFunction(description = "Fills the byte array with 'Count' elements and fills them with 'Value'.")
    public void Fill(int Count, int Value) {
        if (Value > 255 || Value < 0) {
            form.ErrorOccurred(this, "Fill", 17200, "Value out of range.");
            return;
        }
        if (Value > 127)
            Value -= 256;
        bytes = new ArrayList<>(Count);
        for (int i = 0; i < Count; i++)
            bytes.add((byte) Value);
        readIndex = 0;
    }

    public static int toUnsignedInt(byte x) {
        return ((int) x) & 0xff;
    }

    @SimpleFunction(description = "Returns the array as a hex string.")
    public String ToString() {
        StringBuilder sb = new StringBuilder();
        for (byte b : bytes) {
            int i = toUnsignedInt(b);
            sb.append(' ');
            sb.append(prefix);
            sb.append(String.format("%02X", b));
        }
        String s = sb.toString();
        if (s.length() > 0)
            return sb.toString().substring(1);
        else
            return s;
    }

    @SimpleFunction(description = "Appends a byte to the array.")
    public void AddByte(int Byte) {
        if (Byte > 255 || Byte < 0) {
            form.ErrorOccurred(this, "Addbyte", 17200, "Value out of range.");
            return;
        }
        if (Byte > 127)
            Byte -= 256;
        bytes.add((byte) Byte);
    }

    @SimpleFunction(description = "Appends a word (16 Bit) to the array.")
    public void AddWord(int Word) {
        if (Word > 65535 || Word < 0) {
            form.ErrorOccurred(this, "AddWord", 17200, "Value out of range.");
            return;
        }
        if (Word > 32767)
            Word -= 65536;
        byte lsb = (byte) (Word & 0xFF);
        byte msb = (byte) (Word >> 8);
        if (msbFirst) {
            bytes.add(msb);
            bytes.add(lsb);
        } else {
            bytes.add(lsb);
            bytes.add(msb);
        }
    }

    @SimpleFunction(description = "Appends a double word (32 Bit) to the array.")
    public void AddDWord(long DWord) {
        if (DWord > 4294967295L || DWord < 0) {
            form.ErrorOccurred(this, "AddDWord", 17200, "Value out of range.");
            return;
        }
        if (DWord > 2147483647L)
            DWord -= 4294967296L;
        byte b1 = (byte) (DWord & 0xFF); // LSB
        byte b2 = (byte) (DWord >> 8);
        byte b3 = (byte) (DWord >> 16);
        byte b4 = (byte) (DWord >> 24);
        if (msbFirst) {
            bytes.add(b4);
            bytes.add(b3);
            bytes.add(b2);
            bytes.add(b1);
        } else {
            bytes.add(b1);
            bytes.add(b2);
            bytes.add(b3);
            bytes.add(b4);
        }
    }

    @SimpleFunction(description = "Appends a string to the array. US-ASCII codes (7 bit) of the characters are added.")
    public void AddASCIIString(String Text) {
        byte[] b = Text.getBytes(StandardCharsets.US_ASCII);
        for (int i = 0; i < b.length; i++)
            bytes.add(b[i]);
    }

    @SimpleFunction(description = "Appends a string to the array. UTF-8 codes of the characters are added.")
    public void AddUTF8String(String Text) {
        byte[] b = Text.getBytes(StandardCharsets.UTF_8);
        for (int i = 0; i < b.length; i++)
            bytes.add(b[i]);
    }

    @SimpleFunction(description = "Returns the number of bytes to store an UTF-8 coded string.")
    public int GetUTF8ByteSize(String Text) {
        byte[] b = Text.getBytes(StandardCharsets.UTF_8);
        return b.length;
    }

    int getByte(int index) {
        int result = bytes.get(index);
        if (result < 0)
            result += 256;
        return result;
    }

    @SimpleFunction(description = "Gets the byte at the given index in the array. The first array item is at index 'Base'.")
    public int GetByteAt(int Index) {
        Index = Index - base;
        if (Index < 0 || Index >= bytes.size()) {
            form.ErrorOccurred(this, "GetByteAt", 17201, "Index out of range.");
            return 0;
        }
        return getByte(Index);
    }

    @SimpleFunction(description = "Sets the byte at the given index in the array. The first array item is at index 'Base'.")
    public void SetByteAt(int Index, int Byte) {
        if (Byte > 255 || Byte < 0) {
            form.ErrorOccurred(this, "SetByteAt", 17200, "Value out of range.");
            return;
        }

        Index = Index - base;
        if (Index < 0 || Index >= bytes.size()) {
            form.ErrorOccurred(this, "SetByteAt", 17201, "Index out of range.");
            return;
        }
        if (Byte > 127)
            Byte -= 256;
        bytes.set(Index, (byte) Byte);
    }

    @SimpleFunction(description = "Sets the Word at the given index in the array. The first array item is at index 'Base'.")
    public void SetWordAt(int Index, int Word) {
        if (Word > 65535 || Word < 0) {
            form.ErrorOccurred(this, "SetWordAt", 17200, "Value out of range.");
            return;
        }

        Index = Index - base;
        if (Index < 0 || Index >= bytes.size() - 1) { // Für ein Word werden 2 Bytes benötigt.
            form.ErrorOccurred(this, "SetWordAt", 17201, "Index out of range.");
            return;
        }
        if (Word > 32767)
            Word -= 65536;
        byte lsb = (byte) (Word & 0xFF);
        byte msb = (byte) (Word >> 8);
        if (msbFirst) {
            bytes.set(Index, msb);
            bytes.set(Index + 1, lsb);
        } else {
            bytes.set(Index, lsb);
            bytes.set(Index + 1, msb);
        }
    }

    @SimpleFunction(description = "Sets the DWord at the given index in the array. The first array item is at index 'Base'.")
    public void SetDWordAt(int Index, long DWord) {
        if (DWord > 4294967295L || DWord < 0) {
            form.ErrorOccurred(this, "SetDWordAt", 17200, "Value out of range.");
            return;
        }

        Index = Index - base;
        if (Index < 0 || Index >= bytes.size() - 3) { // Für ein DWord werden 4 Bytes benötigt.
            form.ErrorOccurred(this, "SetWordAt", 17201, "Index out of range.");
            return;
        }

        if (DWord > 2147483647L)
            DWord -= 4294967296L;
        byte b1 = (byte) (DWord & 0xFF); // LSB
        byte b2 = (byte) (DWord >> 8);
        byte b3 = (byte) (DWord >> 16);
        byte b4 = (byte) (DWord >> 24);
        if (msbFirst) {
            bytes.set(Index, b4);
            bytes.set(Index + 1, b3);
            bytes.set(Index + 2, b2);
            bytes.set(Index + 3, b1);
        } else {
            bytes.set(Index, b1);
            bytes.set(Index + 1, b2);
            bytes.set(Index + 2, b3);
            bytes.set(Index + 3, b4);
        }
    }

    @SimpleFunction(description = "Appends the content of a ByteArray.")
    public void Append(Component Array) {
        if (!(Array instanceof UrsAI2ByteArray)) {
            form.ErrorOccurred(this, "Append", 17202, "Invalid Type.");
            return;
        }

        for (Byte b : ((UrsAI2ByteArray) Array).bytes)
            bytes.add(b);
    }

    @SimpleFunction(description = "Converts a number to a hexadecimal string.")
    public String ToHex(long Value, int Digits) {

        if (Digits != 2 && Digits != 4 && Digits != 8)
            return "--";

        if (Value < 0 || Value > 4294967295L)
            return "--";

        if (Digits == 2) {
            byte b = (byte) (Value & 0xFF);
            return prefix + String.format("%02X", b);
        }
        if (Digits == 4) {
            int s = (int) (Value & 0xFFFF);
            return prefix + String.format("%04X", s);
        }
        if (Digits == 8) {
            long s = (long) (Value & 0xFFFFFFFF);
            return prefix + String.format("%08X", s);
        }
        String format = "%0" + Digits + "X";

        return prefix + String.format(format, Value);
    }

    @SimpleFunction(description = "Reads sequentially from the array.")
    public int ReadByte() {
        if (readIndex >= bytes.size()) {
            form.ErrorOccurred(this, "Append", 17203, "Read beyond end of array.");
            return 0;
        }
        return getByte(readIndex++);
    }

    @SimpleFunction(description = "Reads sequentially from the array.")
    public int ReadWord() {
        int result = 0;
        if (readIndex >= bytes.size() - 1) {
            form.ErrorOccurred(this, "Append", 17203, "Read beyond end of array.");
            readIndex = bytes.size();
            return 0;
        }
        if (msbFirst) {

            result = (((int) bytes.get(readIndex++)) << 8) & 0xFF00;
            result |= (((int) bytes.get(readIndex++))) & 0xFF;
        } else {
            result = (((int) bytes.get(readIndex++))) & 0xFF;
            result |= (((int) bytes.get(readIndex++)) << 8) & 0xFF00;
        }

        if (result < 0)
            result += 65536;
        return result;
    }

    @SimpleFunction(description = "Reads sequentially from the array.")
    public long ReadDWord() {
        long result = 0;
        if (readIndex >= bytes.size() - 3) {
            form.ErrorOccurred(this, "Append", 17203, "Read beyond end of array.");
            readIndex = bytes.size();
            return 0;
        }
        if (msbFirst) {
            result = (((long) bytes.get(readIndex++)) << 24) & 0xFF000000L;
            result |= (((long) bytes.get(readIndex++)) << 16) & 0xFF0000L;
            result |= (((long) bytes.get(readIndex++)) << 8) & 0xFF00L;
            result |= (((long) bytes.get(readIndex++))) & 0xFFL;
        } else {
            result = (((long) bytes.get(readIndex++))) & 0xFFL;
            result |= (((long) bytes.get(readIndex++)) << 8) & 0xFF00L;
            result |= (((long) bytes.get(readIndex++)) << 16) & 0xFF0000L;
            result |= (((long) bytes.get(readIndex++)) << 24) & 0xFF000000L;
        }
        if (result < 0)
            result += 4294967296L;
        return result;
    }

    @SimpleFunction(description = "Reads sequentially an UTF8 string from the current position to the end.")
    public String ReadUTF8String() {
        byte[] b = new byte[bytes.size() - readIndex];
        int ind = 0;
        while (readIndex < bytes.size())
            b[ind++] = bytes.get(readIndex++);

        return new String(b, StandardCharsets.UTF_8);
    }

    @SimpleFunction(description = "Reads sequentially an UTF8 string from the current position until the delimitation character / string is found.")
    public String ReadUTF8StringUntil(String Delimiter) {
        int temp = readIndex;

        Delimiter = Delimiter.replace("\\0", "\0");


        byte[] b = new byte[bytes.size() - readIndex];
        int ind = 0;
        while (readIndex < bytes.size())
            b[ind++] = bytes.get(readIndex++);

        String s = new String(b, StandardCharsets.UTF_8);
        int pos = s.indexOf(Delimiter);

        if (pos < 0) // not found
            return s;

        s = s.substring(0, pos);
        readIndex = temp + GetUTF8ByteSize(s) + GetUTF8ByteSize(Delimiter);

        return s;
    }

    @SimpleFunction(description = "Reads sequentially an US-ASCII (7 bits) string from the current position to the end.")
    public String ReadASCIIString() {
        byte[] b = new byte[bytes.size() - readIndex];
        int ind = 0;
        while (readIndex < bytes.size())
            b[ind++] = bytes.get(readIndex++);

        return new String(b, StandardCharsets.US_ASCII);
    }

    @SimpleFunction(description = "Reads sequentially an US-ASCII (7 bits) string from the current position until the delimitation character / string is found.")
    public String ReadASCIIStringUntil(String Delimiter) {
        int temp = readIndex;

        Delimiter = Delimiter.replace("\\0", "\0");

        byte[] b = new byte[bytes.size() - readIndex];
        int ind = 0;
        while (readIndex < bytes.size())
            b[ind++] = bytes.get(readIndex++);

        String s = new String(b, StandardCharsets.US_ASCII);
        int pos = s.indexOf(Delimiter);

        if (pos < 0) // not found
            return s;

        s = s.substring(0, pos);
        readIndex = temp + s.length() + Delimiter.length();

        return s;
    }

    @SimpleFunction(description = "Removes the byte at the specified position.")
    public int RemoveByteAt(int Index) {
        Index = Index - base;
        if (Index < 0 || Index >= bytes.size()) {
            form.ErrorOccurred(this, "RemoveByteAt", 17201, "Index out of range.");
            return 0;
        }
        int result = getByte(Index);
        bytes.remove(Index);
        if (readIndex > Index)
            readIndex--;
        return result;
    }

    @SimpleFunction(description = "Removes Count bytes beginning with the current position.")
    public void RemoveBytes(int Count) {

        int toIndex = readIndex + Count;
        if (toIndex >= bytes.size())
            toIndex = bytes.size() - 1;

        bytes.subList(readIndex, toIndex).clear();
    }

    @SimpleFunction(description = "Inserts the byte at the specified position.")
    public void InsertByteAt(int Index, int Byte) {
        Index = Index - base;
        if (Index < 0 || Index > bytes.size()) {
            form.ErrorOccurred(this, "InsertByteAt", 17201, "Index out of range.");
            return;
        }

        if (Byte > 255 || Byte < 0) {
            form.ErrorOccurred(this, "InsertByteAt", 17200, "Value out of range.");
            return;
        }

        bytes.add(Index, (byte) Byte);
        if (readIndex >= Index)
            readIndex++;
        return;
    }

    /**
     * Kopiert den Inhalt in byte[]
     * 
     * @return
     */
    public byte[] toByteArray() {
        byte[] data = new byte[bytes.size()];
        for (int i = 0; i < bytes.size(); i++)
            data[i] = bytes.get(i);
        return data;
    }

    @SimpleFunction(description = "Writes the content of the byte array to a file asynchronously.")
    public void WriteToFile(final String FileName, final boolean Append) {
        WriteToFile(FileName, Append, true);
    }

    @SimpleFunction(description = "Writes the content of the byte array to a file synchronously.")
    public void WriteToFileSync(final String FileName, final boolean Append) {
        WriteToFile(FileName, Append, false);
    }

    // Schreibt die Daten
    private void WriteToFile(final String FileName, final boolean Append, final boolean async) {
        if (FileName.startsWith("//")) {
            form.dispatchErrorOccurredEvent(this, "WriteToFile", ErrorMessages.ERROR_CANNOT_WRITE_ASSET,
                    FileName);
            return;
        }
        if (FileName.startsWith("/")) {
            FileUtil.checkExternalStorageWriteable(); // Only check if writing to sdcard
        }
        try {
            new FileBinaryWriteOperation(form, this, "WriteToFile", FileName, scope, Append, async) {
                String filePath = "";

                @Override
                public void processFile(ScopedFile scopedFile) {
                    java.io.File file = scopedFile.resolve(form);
                    filePath = file.getAbsolutePath();

                    if (!file.exists()) {
                        boolean success = false;
                        try {
                            IOUtils.mkdirs(file);
                            success = file.createNewFile();
                            file.setReadable(true, false);
                            file.setWritable(true, false);
                        } catch (IOException e) {
                            Log.e(LOG_TAG, "Unable to create file " + file.getAbsolutePath());
                        }
                        if (!success) {
                            form.dispatchErrorOccurredEvent(UrsAI2ByteArray.this, method,
                                    ErrorMessages.ERROR_CANNOT_CREATE_FILE, file.getAbsolutePath());
                            return;
                        }
                    }

                    super.processFile(scopedFile);
                }

                @Override
                public boolean processII(OutputStream out) throws IOException {
                    out.write(toByteArray());
                    out.flush();
                    if (async) {
                        form.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                AfterFileWritten(FileName, filePath);
                            }
                        });
                    }
                    return true;
                }

                @Override
                public void onError(IOException e) {
                    super.onError(e);
                    form.dispatchErrorOccurredEvent(UrsAI2ByteArray.this, method,
                            ErrorMessages.ERROR_CANNOT_WRITE_TO_FILE,
                            getFile().getAbsolutePath());
                }
            }.run();
        } catch (StopBlocksExecution e) {
            // This is okay because the block is designed to be asynchronous.
        }
    }

    /**
     * Event indicating that the contents of the file have been written.
     *
     * @param fileName the name of the written file
     */
    @SimpleEvent(description = "Event indicating that the contents of the file have been written.")
    public void AfterFileWritten(String FileName, final String FilePath) {
        // invoke the application's "AfterFileSaved" event handler.
        EventDispatcher.dispatchEvent(this, "AfterFileWritten", FileName, FilePath);
    }

    // Copy from File.java:

    /**
     * Deletes a file from storage. Prefix the `fileName`{:.text.block} with `/` to
     * delete a specific
     * file in the SD card (for example, `/myFile.txt` will delete the file
     * `/sdcard/myFile.txt`).
     * If the `fileName`{:.text.block} does not begin with a `/`, then the file
     * located in the
     * program's private storage will be deleted. Starting the
     * `fileName`{:.text.block} with `//` is
     * an error because asset files cannot be deleted.
     *
     * @param fileName the file to be deleted
     */
    @SimpleFunction(description = "Deletes a file from storage. Prefix the filename with / to "
            + "delete a specific file in the SD card, for instance /myFile.txt. will delete the file "
            + "/sdcard/myFile.txt. If the file does not begin with a /, then the file located in the "
            + "programs private storage will be deleted. Starting the file with // is an error "
            + "because assets files cannot be deleted.")
    public void DeleteFile(final String FileName) {
        if (FileName.startsWith("//")) {
            form.dispatchErrorOccurredEvent(this, "Delete",
                    ErrorMessages.ERROR_CANNOT_DELETE_ASSET, FileName);
            return;
        }
        try {
            new FileWriteOperation(form, this, "Delete", FileName, scope, false, true) {
                @Override
                public void processFile(ScopedFile scopedFile) {
                    java.io.File file = scopedFile.resolve(form);
                    // Invariant: After deleting, the file should not exist. If the file already
                    // doesn't exist, mission accomplished!
                    if (file.exists() && !file.delete()) {
                        form.dispatchErrorOccurredEvent(UrsAI2ByteArray.this, "Delete",
                                ErrorMessages.ERROR_CANNOT_DELETE_FILE, FileName);
                    }
                }
            }.run();
        } catch (StopBlocksExecution e) {
            // This is okay because the block is designed to be asynchronous.
        }
    }

    /**
     * Reads text from a file in storage. Prefix the `fileName`{:.text.block} with
     * `/` to read from a
     * specific file on the SD card (for example, `/myFile.txt` will read the file
     * `/sdcard/myFile.txt`). To read assets packaged with an application (also
     * works for the
     * Companion) start the `fileName`{:.text.block} with `//` (two slashes). If a
     * `fileName`{:.text.block} does not start with a slash, it will be read from
     * the application's
     * private storage (for packaged apps) and from `/sdcard/AppInventor/data` for
     * the Companion.
     *
     * @param fileName the file from which the text is read
     */
    @SimpleFunction(description = "Reads text from a file in storage asynchronously. "
            + "Prefix the filename with / to read from a specific file on the SD card. "
            + "for instance /myFile.txt will read the file /sdcard/myFile.txt. To read "
            + "assets packaged with an application (also works for the Companion) start "
            + "the filename with // (two slashes). If a filename does not start with a "
            + "slash, it will be read from the applications private storage (for packaged "
            + "apps) and from /sdcard/AppInventor/data for the Companion.")
    public void ReadFromFile(final String FileName) {
        ReadFromFile(FileName, true);
    }

    @SimpleFunction(description = "Reads text from a file in storage synchronously. "
            + "Prefix the filename with / to read from a specific file on the SD card. "
            + "for instance /myFile.txt will read the file /sdcard/myFile.txt. To read "
            + "assets packaged with an application (also works for the Companion) start "
            + "the filename with // (two slashes). If a filename does not start with a "
            + "slash, it will be read from the applications private storage (for packaged "
            + "apps) and from /sdcard/AppInventor/data for the Companion.")
    public void ReadFromFileSync(final String FileName) {
        ReadFromFile(FileName, false);
    }

    private void ReadFromFile(final String FileName, boolean async) {
        bytes.clear();

        try {
            new FileReadOperation(form, this, "ReadFromFile", FileName, scope, async) {
                @Override
                public boolean process(byte[] contents) {
                    for (int i = 0; i < contents.length; i++) {
                        bytes.add(contents[i]);
                    }

                    if (async) {
                        form.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                AfterFileRead();
                            }
                        });
                    }
                    return true;
                }

                @Override
                public void onError(IOException e) {
                    if (e instanceof FileNotFoundException) {
                        Log.e(LOG_TAG, "FileNotFoundException", e);
                        form.dispatchErrorOccurredEvent(UrsAI2ByteArray.this, "ReadFromFile",
                                ErrorMessages.ERROR_CANNOT_FIND_FILE, FileName);
                    } else {
                        Log.e(LOG_TAG, "IOException", e);
                        form.dispatchErrorOccurredEvent(UrsAI2ByteArray.this, "ReadFromFile",
                                ErrorMessages.ERROR_CANNOT_READ_FILE, FileName);
                    }
                }
            }.run();
        } catch (StopBlocksExecution e) {
            Log.d(LOG_TAG, e.toString());
            // This is okay because the block is designed to be asynchronous.
        }
    }

    /**
     * Event indicating that the contents from the file have been read.
     *
     * @param text read from the file
     */
    @SimpleEvent(description = "Event indicating that the contents from the file have been read.")
    public void AfterFileRead() {
        // invoke the application's "GotText" event handler.
        EventDispatcher.dispatchEvent(this, "AfterFileRead");
    }

    /**
     * Specifies the default scope for files accessed using the File component. The
     * App scope should
     * work for most apps. Legacy mode can be used for apps that predate the newer
     * constraints in
     * Android on app file access.
     *
     * @param scope the default file access scope
     */
    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_FILESCOPE, defaultValue = "App")
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Specifies the default scope for files accessed using the File component.", userVisible = false)
    public void DefaulFiletScope(FileScope scope) {
        this.scope = scope;
    }

    /**
     * Indicates the current scope for operations such as ReadFrom and SaveFile.
     *
     * @param scope the target scope
     */
    @SimpleProperty
    public void Scope(FileScope scope) {
        this.scope = scope;
    }

    @SimpleProperty
    public FileScope Scope() {
        return scope;
    }
}